/* Copyright (c) 2004, 2012, Oracle and/or its affiliates. 
All rights reserved. */

/*
 *************************************************************************
 *   NAME
 *      clsagfw.h - Programming interface (C API) to develop CRS HA
 *                  Agents.
 *   DESCRIPTION
 *      These APIs facilitates CRS(Cluster Ready Services) users to develop
 *      HA Agents which helps users to plugin thier applications/resoruces 
 *      into CRS HA framework.
 *
 *      The Agent Framework aims to provide improved communication guarantees 
 *      and provide for lifetime monitoring of applications. It facilitates 
 *      administrators to develop their own custom agents with very minimal
 *      effort. The Agent Framework abstracts all the CRSD communication 
 *      details from administrators and provides a clean interface 
 *      (a.k.a resource entry points) for a CRS daemon to deal with resources.
 *      Agent developers can use these entry points to plug-in the required 
 *      functionality for a specific resource, with respect to how to start
 *      a resource etc. Administrators can implement entry points in 
 *      various ways: shell scripts/perl scripts/C/C++ functions/Java functions. 
 *      Agent Framework will have the following entry points:
 *
 *      **START**: The start (online) entry point will have the functionality
 *      to bring the resource online. Agent Framework calls this entry point
 *      whenever it receives an ONLINE command from CRSD for a resource.
 *
 *      **STOP**: The stop (offline) entry point will have the functionality 
 *      to bring down the resource. Agent Framework will call this entry point 
 *      whenever it receives OFFLINE command from CRSD for a resource. 
 *
 *      **CHECK**: The check (monitor) entry point will have the functionality 
 *      to monitor the health of a resource. Agent Framework will call this 
 *      entry point periodically, if it notices any state changes it will notify
 *      CRSD about the change in the state of the specific resource. 
 *
 *      **CLEAN**: The clean entry point is executed automatically by the 
 *      Agent Framework, when the specific resource has to be taken offline 
 *      and all the current actions must be terminated.
 *
 *      **DISCOVERY**: Discovery entry point will be called by CRSD, whenever
 *      it needs to discover dynamic attributes for a specific resource.
 *
 *      **MODIFY**: (Since CRS 12.1) Agent Framework will call this entrypoint 
 *      on online resources whenever CRSD modifies resource attributes, allowing 
 *      the agent to make any necessary updates.
 *
 *      **DELETE**: (Since CRS 12.1) The delete entrypoint will have 
 *      funcionality to delete a resource. Agent Framework will call this entry
 *      point to allow agents to perform final clean up tasks.
 *
 *      **ACTION**: (Since CRS 12.1) The action entrypoint will have
 *      funcionality for custom actions. Actions must be registered with the 
 *      resource, and can only use resource attributes as inputs. Agent
 *      Framework calls this entrypoint whenever it receives an ACTION command
 *      from CRSD for a resource.
 *      
 *   EXPORTED APIS
 *      clsagfw_init
 *      clsagfw_add_type
 *      clsagfw_set_entrypoint
 *      clsagfw_startup
 *      clsagfw_log
 *      clsagfw_send_status
 *      clsagfw_modify_attribute
 *      clsagfw_exit
 *      clsagfw_get_attrvalue
 *      clsagfw_set_cookie
 *      clsagfw_get_cookie
 *      clsagfw_delete_cookie
 *      clsagfw_set_exitcb
 *      clsagfw_get_cmdid
 *      clsagfw_get_resource_name
 *      clsagfw_get_resource_id
 *      clsagfw_get_type_name
 *      clsagfw_is_cmd_timedout
 *      clsagfw_set_resource_state_label
 *      clsagfw_check_resource
 *      clsagfw_get_check_type
 *
 *   NOTES
 *   (1) Context initialization and persistence:
 *       The agent framework is initialized by the clsagfw_init function call.
 *       There is no context that needs to be maintained by the users.
 *       Action entry functions and various other API calls accept a context
 *       (clsagfw_aectx) as a parameter. However, this is an opaque data structure
 *       that is maintained by the agent framework and need not be initialized or 
 *       destroyed by the agent. This context is specific to the action entry 
 *       point being called and cannot be used across calls.
 *       clsagfw_exit performs an implicit cleanup of all internal structures.
 *       
 *   (2) Threading support:
 *       The agent framework internally generates multiple threads for command 
 *       processing and execution. At any point, only one command is executed
 *       for a resource, subject to a timeout.   
 *    
 *   (3) Callbacks:
 *       All of the agent's functionality is implemented with callbacks. The agent 
 *       implements functions of type clsagfw_action_entry and registers them
 *       with the framework using the clsagfw_set_entrypoint function. 
 *       Callbacks need to be specified for the START, STOP, CLEAN and CHECK actions.
 *       If any of these action entry points are not defined, it is assumed
 *       that the entry point is defined in a script.
 *       Cleanup before the final shutdown can be handled by registering an exit callback
 *       function using clsagfw_set_exitcb.
 *
 *   (4) Resource Considerations:
 *       All internal resources are cleaned up by the framework during agent exit. If 
 *       there are any agent specific resources that need to be cleaned up, a callback
 *       needs to be registered using clsagfw_set_exitcb.
 *
 *   (5) Error Handling:
 *       The agent framework function APIs return status code of type clsagfwret,
 *       which can be tested for success or failure.
 *       All action entry callback functions need to return CLSAGFW_AE_SUCCESS or
 *       CLSAGFW_AE_FAIL depending on whether the function succeeded in its task or
 *       not. This is used internally by the framework to determine the state of the
 *       resource. Refer to the clsagfw_action_entry definition for more information.
 *
 *   (6) CLSCRS API notes:
 *       When using clscrs APIs inside the agent entry points, there
 *       are a few considerations to keep in mind:
 *       1. Some actions (start/stop/clean) are executed under a lock on
 *          the resource instance. Thus, issuing a request to the server to act
 *          on the resource directly or by extension of a relation will be a
 *          dead-lock.
 *       2. Issuing read-only commands (clscrs_stat2) is generally safe UNLESS 
 *          it's an "initial check", where the agent must not call back on the 
 *          server as that will dead-lock as well. Check type can be queried 
 *          through clsagfw APIs.
 *   
 *   (7) Version Support:
 *       All APIs and entrypoints are available as of 11g Release 2, unless 
 *       specifically noted.
 *
 *   RELATED DOCUMENTS
 *   (1) Oracle Clusterware Adminstration and Deployment Guide - 11g Release 2
 *   (2) Oracle Technology Network - http://www.oracle.com/technology
 *
 **************************************************************************
 */

#ifndef CLSAGFW_ORACLE
#define CLSAGFW_ORACLE

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#ifndef ORATYPES
# include <oratypes.h>
#endif
#include <stdarg.h>

#define AGFWEXPORT

/*
 * Following are the return codes from the CLS AGFW API 
 */
typedef enum clsagfwret
{
    CLSAGFW_SUCCESS    = 0,
    CLSAGFW_FAIL       = 1
}clsagfwret;

/*
 * Return value from AGFW entry point(except CHECK entry point).
 */
typedef enum clsagfw_ae_ret
{
  /*
   * Action entry execution finished successfully.
   */
  CLSAGFW_AE_SUCCESS = 0,

  /*
   * Action entry exection failed
   */
  CLSAGFW_AE_FAIL,

  /*
   * Action entry timed out
   */
  CLSAGFW_AE_TIMEDOUT
}clsagfw_aeret;

 /*
  * At present CHECK is the only action entry which returns 
  * the resource status, rest of the action entries should return 
  * execution status.
  */
typedef enum clsagfw_res_status
{
  /**
   * Check action can return ONLINE state if the resource is 
   * in functioning state. Once resource is in this state 
   * AGFW would monitor the reource.
   */
  CLSAGFW_ONLINE = 0,

  /**
   * Two ways to indicate a resources's being offline exist: planned
   * and unplanned. Please follow these rules when deciding on which
   * value is appropriate. When the resource state transitions are
   * originated by or though the crsd, then it's assumed that the
   * intent for this resource is to be offline (TARGET = OFFLINE)
   * irrespective of which value is returned from the check action.
   * However, when an agent detects that the state of a resource 
   * has changed though an unsolited by crsd sequencer of events (for
   * example, somebody stopped the resource through a non-crsd
   * interface), then the intent needs to be carried over from the
   * agent to the crsd. The intent will then be the deciding factor
   * for the following
   * - Keeping or changing the value of the resource's TARGET attribute 
   *   (PLANNED_OFFLINE will indicate the target needs to be changed
   *   to OFFLINE, where as UNPLANNED_OFFLINE will not change the target)
   * - Leaving the resource OFFLINE or attempting to conduct resource
   *   recovery such as restarting locally or failover the resource
   *   over to a different server in the cluster. (PLANNED_OFFLINE
   *   will make crsd leave the resource as is; UNPLANNED_OFFLINE will
   *   be interpreted as a sign to commence resource recovery)
   */
  CLSAGFW_UNPLANNED_OFFLINE,
  CLSAGFW_PLANNED_OFFLINE,

  /**
   * Check action can return UNKNOWN state, if it is not able find what
   * state the resource is in. In response to this state PE would not
   * attempt any failover/restart of the resource. AGFW would continue 
   * monitoring the resource if the resource's previous state is
   * either ONLINE or PARTIAL.
   */
  CLSAGFW_UNKNOWN,

  /**
   * Check can return this state when it knows that resource is
   * partially ONLINE and some of its services are available.  PE
   * considers this state as partially ONLINE hence it would 
   * not attempt any failover/restart of the resource. 
   * AGFW would monitor the resource in this state.
   */
  CLSAGFW_PARTIAL,

  /**
   * Check can return this state when  ever it detects that
   * resource is not in functioning state and some of its components
   * are failed and some of the components are in running state. So
   * some clean up is required to restart the resource. In response to
   * this state PE would call CLEAN action to clean up the resource.
   * So at the end of the clean action, resource state is expected to 
   * be OFFLINE. And then based on resource's policy PE may attempt to
   * restart/faiover the resource
   * Under no circumstances AGFW would monitor the failed resources.
   */
  CLSAGFW_FAILED,

  /** 
   * END OF MARKER FOR FRAMEWORK RESOURCE STATUS.
   * NO RESOURCE STATUS SHOULD BE DEFINED AFTER THIS MARKER.
   */
  CLSAGFW_RESSTATE_EOM
}clsagfw_res_status;


/*************************************************
 * Agent Framework  action entry points:
 *
 *************************************************/
#define CLSAGFW_ACTION_CODE         0x0101
typedef enum {

    /**
     * CLSAGFW_ACTION_START: The start(online) entry point 
     * will have the functionality to bring up the resource
     * online. Agent Framework calls this entry point whenever
     * it receives online message from CRSD
     */
    CLSAGFW_ACTION_START  = CLSAGFW_ACTION_CODE,

    /**
     * CLSAGFW_ACTION_STOP: The stop(offline) entry point will 
     * have the functionality to bring down the resource. 
     * Agent Framework will call this entry point whenever 
     * it receives offline message from CRSD.
     */
    CLSAGFW_ACTION_STOP,

    /**
     * CLSAGFW_ACTION_CLEAN: 
     * The clean entry point is executed when ever there is a need to
     * clean up the application. It is a non-graceful operation,
     * users must forcefully terminate their application and cleanup
     * the application specific environment, so that it can be restarted.
     */
    CLSAGFW_ACTION_CLEAN,

    /**
     * CLSAGFW_ACTION_CHECK: The check(monitor) entry point 
     * will have the functionality to monitor the health
     * of a resource. Agent Framework will call this
     * entry point periodically, if it notices any state 
     * changes it will notify CRSD about that resource state.
     */
    CLSAGFW_ACTION_CHECK,

    /**
     * CLSAGFW_ACTION_ABORT: If C entry point hangs or times out 
     * Agent Framework calls this function. 
     * If user does not supply an abort function then framework
     * calls its default abort function.
     */
    CLSAGFW_ACTION_ABORT,

    /**
     * CLSAGFW_ACTION_RES_DELETED: If the resource is about to be deleted,
     * Agent Framework calls this function to allow the resource to
     * fun any cleanup tasks before deletion.
     */
    CLSAGFW_ACTION_RES_DELETED,

    /**
     * CLSAGFW_RESATTR_MODIFIED: When ever there is a change in
     * resource attribute, AgentFramework invokes this entry point
     * with new and old values of that attribute.
     */
    CLSAGFW_RESATTR_MODIFIED,

    /**
     * CLSAGFW_RES_ACTION: This entrypoint allows a limited set of custom
     * actions. Action names must be preregistered with the resource, and
     * inputs must be resource attributes. When this entrypoint is invoked,
     * the specific action name will also be passed to the agent.
     */
    CLSAGFW_RES_ACTION,

    /**
     * END OF MARKER FOR FRAMEWORK ACTIONS.
     * NOT ACTION SHOULD BE DEFINED AFTER THIS MARKER.
     */
    CLSAGFW_ACTION_EOM
} clsagfw_aecode; /* -- enum AGFW_ACTION_CODES -- */

/**
 * Agent Framework's Action entry context. When ever framework 
 * invokes an user entry point function, this context would be 
 * passed as an argument. Some of the agent framework API's 
 * expects this context.
 */
typedef struct clsagfw_aectx    clsagfw_aectx;

/**
 * Prototype for Agent Framework's action entry point
 *
 * Arguments:
 *      aectx[IN]:
 *      pointer to action entry context. This context is
 *      passed by the framework to the entry point.
 *
 * Return Value:
 *      On successful completion of the entry point tasks, it 
 *      should return CLSAGFW_AE_SUCCESS otherwise it should return
 *      CLSAGFW_AE_FAIL
 *      If action entry is CLSAGFW_ACTION_CHECK then return value is
 *      the state of the resource (clsagfw_res_state).
 */
typedef ub4 (*clsagfw_action_entry) (const clsagfw_aectx *aectx);

/*
 * Status value codes that are passed to the AGFW exit function.
 * These codes can be processed by the user's exit callback function
 * to determine why the shutdown occurred.
 */
typedef enum clsagfw_exit_status
{
  /*
   * Unable to create the agent framework
   */
  CLSAGFW_EXIT_INITFAIL = -1,

  /*
   * No errors seen, normal shutdown due to explicit call to AGFW exit.
   */
  CLSAGFW_EXIT_SUCCESS = 0,

  /*
   * No resources left to monitor, agent shutting itself down.
   */
  CLSAGFW_EXIT_AUTOSHUTDOWN = 1,

  /*
   * Agent disconnected from server.
   */
  CLSAGFW_EXIT_DISCONNECT_FROM_SERVER = 2,

  /*
   * Exception while trying to execute user's action entry point
   */
  CLSAGFW_EXIT_USER_EXCEPTION = 3,

  /*
   * Abort command failed, since no valid abort entry point was defined.
   */
  CLSAGFW_EXIT_ABORT_FAILED = 4,

  /*
   * Internal error occurred in the agent framework.
   */
  CLSAGFW_EXIT_INTERNALERROR = 5,

  /*
   * Maximum enum value, for compile-time control on enum size
   */
  CLSAGFW_EXIT_MAXVAL = SB4MAXVAL
} clsagfw_exitcode;

/**
 * When ever agent framework shutting down (graceful shutdown)
 * it invokes this call back function. Agent developers can 
 * use this call back to perform any necessary cleanup.
 */
typedef void (*clsagfw_exit_callback) (sb4 exitCode);
AGFWEXPORT void clsagfw_set_exitcb(clsagfw_exit_callback exit_func);

/**
 * Allowed framework initialization flags for clsagfw_init:
 *
 *  CLSAGFW_FLAGS_CLSD_USESYNC  : Writes to the log files are done synchronously.
 *                                This also makes all calls to writes to be 
 *                                blocking.
 *
 *  CLSAGFW_FLAGS_CLSD_DUPLIO_OK: Higher level buffering may cause duplicate 
 *                                messages, which may not be OK in some cases.
 *                                Use this flag to choose if this is OK.
 *
 *  CLSAGFW_FLAGS_CLSD_USETHREAD: Use dedicated IO thread for logging
 *                                Default is to use THREAD for CLSD.
 *                                This flag is depricated.
 *
 *  CLSAGFW_FLAGS_CLSD_NOTHREAD: No separate thread used for logging.
 *                               Default is to use THREAD for CLSD
 */
#define CLSAGFW_FLAGS_RESERVED_1              ((ub4)0x0001)
#define CLSAGFW_FLAGS_RESERVED_2              ((ub4)0x0002)
#define CLSAGFW_FLAGS_CLSD_USESYNC            ((ub4)0x0004)
#define CLSAGFW_FLAGS_CLSD_DUPLIO_OK          ((ub4)0x0008)

/** Depricated, From 11.2.0.2 AGFW logging uses threads by default */
#define CLSAGFW_FLAGS_CLSD_USETHREAD          ((ub4)0x0010)

#define CLSAGFW_FLAGS_CLSD_NOTHREAD           ((ub4)0x0020)

#define CLSAGFW_FLAGS_RESERVED_3              ((ub4)0x0040)

#define CLSAGFW_FLAGS_RESERVED_4              ((ub4)0x0080)

/**
 * This function initializes the agent frameworks internal 
 * data structures. Agent developers needs to call this 
 * function prior to use any of the Agent Frameworks APIs.
 *
 * Arguments: 
 *      argc[IN]: 
 *          argc passed to the agent program main() function.
 *      argv[IN]: 
 *          argv passed to the agent program main() function.
 *      flags[IN]: 
 *          Framework initialization flags.
 *      logName[IN}:
 *          Logname used for agent process. This is an optional parameter.
 *          By default agent framework uses resource type name as the 
 *          logname with the extension .log. 
 *       reserved[IN]:
 *          reserved for future usage.
 *
 * Return Value: 
 *      On successful initialization it returns CLSAGFW_SUCCESS,
 *      on failure it returns CLSAGFW_FAIL
 */
AGFWEXPORT clsagfwret clsagfw_init(sb4 argc, 
                                   oratext **argv, 
                                   ub4 flags,
                                   const oratext *logName, 
                                   void *reserved);

/**
 * Adds a resource type to the agentframe work. It can be called
 * multiple times to add different types. 
 *
 * Arguments:
 *      type_name[IN]: Name of the resource type.
 *      
 * Return Value: CLSAGFW_SUCCESS OR  CLSAGFW_FAIL
 *
 */
AGFWEXPORT clsagfwret clsagfw_add_type(const oratext *type_name);

/**
 * This API can be used by agent developers to initialize the
 * application specific action entry points.  
 *
 * Arguments:
 *      type_name[IN]: Name of the resource type which has been
 *                       added earlier.
 *      action_entry[IN]: 
 *          Agent's application specific function pointer.
 *          If no action entry is specified for START,STOP 
 *          OR CHECK actions, Framework assumes a script 
 *          located at:
 *          $ORA_CRS_HOME/bin/$AgentName/{start/stop/check}
 *
 *      action_entry_code[IN]:
 *          one of the above mentioned action entry codes.
 *
 * Return Value:  void
 *
 */
AGFWEXPORT void clsagfw_set_entrypoint(const oratext *type_name,
                                       clsagfw_action_entry action_entry, 
                                       clsagfw_aecode action_entrycode);

/**
 * API to start the agent framework
 *
 * Arguments: void 
 *
 * Return Value: On failure this API returns CLSAGFW_FAIL. On success
 * this function never returns.
 */
AGFWEXPORT clsagfwret clsagfw_startup(void);

/**
 * This API can be used by agent developers to print log messages into
 * agent's log file.
 *      ctx[IN]: AgentFramework's context.
 *
 *      log_level: debug level of the log message[1-5].
 *        1 = log all the messages which are marked as 1.
 *        2 = log all the messages which are marked as 1 and 2
 *        3 = log all the messages which are marked as 1,2 and 3
 *        4 = log all the messages which are marked as 1,2,3 and 4
 *        5 = log all the messages which are marked as 1,2,3,4 and 5
 *
 *      fmtp[IN]: message format string.
 *
 * Return Value:  void
 */
AGFWEXPORT void clsagfw_log(const clsagfw_aectx *ctx, 
                            const ub4 log_level,
                            const oratext *fmtp, ...);

/**
 * This API can be used by agent developers to send any
 * error/warning/progress information to crsd clients.
 * Arguments:
 *      ctx[IN]: AgentFramework's context.
 *      type[IN]:
 *             CLSAGFW_STATUS_PROGRESS: Status message is a progress
 *             CLSAGFW_WARNING: Status message is a warning message.
 *             CLSAGFW_STATUS_ERROR: Status message is an error
 *             message.
 *      fmtp[IN]: message format string.
 *
 * Return Value:  void
 */
typedef enum clsagfw_status_type
{
  CLSAGFW_STATUS_PROGRESS   = 0,
  CLSAGFW_STATUS_WARNING    = 1,
  CLSAGFW_STATUS_ERROR      = 2,
}clsagfw_status_type;
AGFWEXPORT void clsagfw_send_status(const clsagfw_aectx* ctx,
                                    clsagfw_status_type  type,
                                    const oratext *fmtp, ...);

/**
 * This API can be used by agent developers in a background
 * thread to send any error/warning/progress information. The 
 * status message will be forwarded to a client if available.
 * Arguments:
 *      ctx[IN]: Agent framework's context. 
 *      pResId[IN]: resource identifier for which the status
 *                  message applies.
 *      type[IN]:
 *             CLSAGFW_STATUS_PROGRESS: Status message is a progress
 *             CLSAGFW_WARNING: Status message is a warning message
 *             CLSAGFW_STATUS_ERROR: Status message is an error
 *             message.
 *      fmtp[IN]: message format string.
 *
 * Return Value:  void
 */
AGFWEXPORT void clsagfw_send_status2(const clsagfw_aectx* ctx,
                                     const oratext*       pResId,
                                     clsagfw_status_type  type,
                                     const oratext*       fmtp, ...);
/**
 * Sends an attribute modification message to CRSD engine.
 *
 * Argumetns:
 *      pAectx[IN]: context which is passed to user entry point.
 *      pAttrName[IN]: name of the attribute to be modified.
 *      pAttrValue[IN]: Value of the attribute (in string format)
 *
 * NOTE:
 *    This API can be used only from START/STOP/CLEAN actions.
 *    It can not be used from any ohter actions.
 */
AGFWEXPORT void
clsagfw_modify_attribute(const clsagfw_aectx *pAeCtx,
                         const oratext *pAttrName, 
                         const oratext *pAttrValue);

/**
 * Framework's exit function. Agent developers can use this function
 * to shutdown the agent when ever they they encounter some fatal
 * errors. This gives the framework an opprtunity to clean its data
 * structures.
 *
 * Arguments:
 *      exitcode[IN}: exit code of the program.
 *
 * Return Value:  void
 */
AGFWEXPORT void clsagfw_exit(sb4 exitCode);


/**
 * This method can be used by agent entry point  to query the 
 * value of of an particular attribute.
 * Arguments:
 *      ctx[IN]: context which is passed to the user entry point.
 *      argName[IN]: name of the argument.
 *      argValue[OUT]: pointer to the argument value.
 *
 * Return Value: CLSAGFW_SUCCESS OR  CLSAGFW_FAIL
 */
AGFWEXPORT clsagfwret clsagfw_get_attrvalue(const clsagfw_aectx *ctx, 
                                            const oratext *argName, 
                                            const oratext **argValue);

/**
 * This method creates an iterator to navigate through all the
 * resource attributes from an agent framework entry point.
 *
 * Arguments:
 *   ctx[IN]  : context which is passed to the user entry point.
 *   flags[IN]: Flag to indicate the type of interested attributes.
 *              (All the attributes OR just modified attributes).
 *
 * ReturnValue:
 *   On successful creation CLSAGFW_SUCCESS other wise CLSAGFW_FAIL.
 *
 * NOTES: 
 *   1. Callers should not manage the memory of the iterator.
 *   2. CLSAGFW_MODIFED_ATTRS only applicable when entrypoint/callback 
 *      is registered with ae code CLSAGFW_RESATTR_MODIFIED.
 *   3. At any given time frame an entry point can create only one
 *      iterator. All the subsequent attempts to create the new iterator
 *      would get the pointer to the iterator which is already created.
 */
typedef enum clsagfw_ai_flags
{
  CLSAGFW_ALL_ATTRS = 0x01,
  CLSAGFW_MODIFIED_ATTRS = 0x02
}clsagfw_ai_flags;

AGFWEXPORT clsagfwret
clsagfw_create_attr_iterator(const clsagfw_aectx *pAeCtx, 
                             clsagfw_ai_flags    flags);

/**
 * This method can be used by to retrieve the attributes from the
 * already created attribute iterator. Every invokation advances the
 * iterator to point to the next avialble attribute.
 *
 * Arguments:
 *   ctx[IN]  : context which is passed to the user entry point.
 *   argName[OUT]: pointer to the name of the argument.
 *   argValue[OUT]: pointer to the argument value.
 *
 * Return Value: 
 *   CLSAGFW_FAIL if iterator reaches the end mark OR if there is any
 *   internal agentfw error.
 *
 *   CLSAGFW_SUCCESS when there is a valid attribute in the iterator.
 */
AGFWEXPORT clsagfwret 
clsagfw_get_attr_from_iterator(const clsagfw_aectx *pAeCtx,
                               const oratext **argName,
                               const oratext **argValue);

/**
 * This method resets the iterator so that subsequent get_attr calls
 * reads the attributes from the begining of the list.
 * Arguments:
 *   ctx[IN]  : context which is passed to the user entry point.
 *   flags[IN]: Flag to indicate the type of interested attributes.
 *              (All the attributes OR just modified attributes).
 * NOTES: 
 *   callers can reset the iterator to fetch different kind of
 *   attributes by specifying the clsagfw_ai_flags.
 */
AGFWEXPORT clsagfwret 
clsagfw_reset_attr_iterator(const clsagfw_aectx *pAeCtx,
                            clsagfw_ai_flags flags);

/**
 * Stores a cookie in Agentframework. Agents can query back this
 * cookie later on.
 * Arguments:
 *    key[IN]     key identifier of the cookie.
 *    Value[IN]   pointer to the value to be stored. 
 * Return Value: 
 *    CLSAGFW_SUCCESS on creating valid cookie otherwise CLSAGFW_FAIL
 *
 */
AGFWEXPORT clsagfwret clsagfw_set_cookie(const oratext *key, const void *value);

/**
 * Gets the value of the stored cookie from agent framework.
 * Arguments:
 *    key[IN]     key identifier of the cookie.
 *    Value[OUT]   pointer to the value retrieved from agfw.
 * Return Value: CLSAGFW_SUCCESS on finding valid cookie otherwise CLSAGFW_FAIL
 */
AGFWEXPORT clsagfwret clsagfw_get_cookie(const oratext *key, const void **value);

/**
 * Delete the stored cookie value from Agent framework.
 * Arguments:
 *    key[IN]     key identifier of the cookie.
 * Return Value: CLSAGFW_SUCCESS on finding valid cookie otherwise CLSAGFW_FAIL
 */
AGFWEXPORT clsagfwret 
clsagfw_delete_cookie(const oratext *key);

/**
 * Returns the action entry code (CLSAGFW_ACTION_START/STOP/CHECK etc..)
 * from the action entry context.
 * Arguments:
 *      pAectx[IN]: context which is passed to user entry point.
 */
AGFWEXPORT clsagfw_aecode 
clsagfw_get_cmdid(const clsagfw_aectx *pAeCtx);

/**
 * Returns the resource name for a given entry point context.
 * Arguments:
 *      pAectx[IN]: context which is passed to user entry point.
 */
AGFWEXPORT const oratext* 
clsagfw_get_resource_name(const clsagfw_aectx *pAeCtx);

/**
 * Returns the Id of the resource for a given entry point context.
 * Arguments:
 *      pAectx[IN]: context which is passed to user entry point.
 */
AGFWEXPORT const oratext* 
clsagfw_get_resource_id(const clsagfw_aectx *pAeCtx);

/**
 * Returns the resource type name for a given entry point context.
 * Arguments:
 *      pAectx[IN]: context which is passed to user entry point.
 */
AGFWEXPORT const oratext* 
clsagfw_get_type_name(const clsagfw_aectx *pAeCtx);

/**
 * Returns true in case command timed out. Can be used in abort
 * function to know whether abort has been called due to internal
 * command time out or  cancel operation from administrator.
 * Arguments:
 *      pAectx[IN]: context which is passed to user entry point.
 */
AGFWEXPORT bool
clsagfw_is_cmd_timedout(const clsagfw_aectx *pAeCtx);

/**
 * Sets a custom label associated with the resource status. 
 * Check entry point can make use of this API to associate a custom
 * label in addition to the resource state. 
 * Arguments:
 *      pAectx[IN]: context which is passed to user entry point.
 *      pLablel[IN]: custom resource state label.
 */
AGFWEXPORT  void
clsagfw_set_resource_state_label(const clsagfw_aectx *pAeCtx, 
                                 const oratext *pLabel);

/**
 * Requests the AGFW to initiate a check on the resource
 * Arguments:
 *    pResId[IN]: id of the resource which needs to be checked.
 *                this can be obtained by clagfw_get_resource_id() api.
 * Returns:
 *    CLSAGFW_SUCCESS : if check action for the resource is initiated/queued
 *    CLSAGFW_FAIL : if resource instance not available or if check is already
 *            in progress. 
 */
AGFWEXPORT clsagfwret
clsagfw_check_resource(const oratext *pResId);

/**
 * Terminates the agent process and sends feedback about the resource
 * that is the cause of the agent exit.
 * Arguments :
 *     rid[IN] : resource that is the cause of the agent exit
 *     exitCode[IN] : process exit code
 *     retry[IN] : whether to restart the problem resource
 *
 * API introduced in 12.1.
 */
AGFWEXPORT void
clsagfw_exit2(const oratext *rid, sb4 exitCode, bool retry);


/**
 * Returns the type of check being invoked. Should be used by CHECK
 * entry point only. At present CHECK is classified as one of the 
 * following:
 *
 * CLSAGFW_INITIAL_CHECK: Check entry point being invoked as part of
 * the initial probe request from the PE when crsd is rebooting/restarting.
 *
 * CLSAGFW_CHECK_AFTER_CMD: Check entry point being invoked to follow 
 * up the prior action (start/stop/clean etc..) to determine
 * the resulted resource state after the action execution.
 *
 * CLSAGFW_PERIODIC_CHECK: Check entry point being invoked as part
 * of the periodical check generated by the agfw. This is governed by
 * the CHECK_INTERVAL.
 *
 * CLSAGFW_EXTERNAL_CHECK: Check entry point being invoked by AGFW
 * when it has received the explicit check request from the PE.
 *
 * CLSAGFW_CHECKFROM_EP: Check action invoked by one of the agfw entry
 * point itself.
 */
typedef enum clsagfw_checktype
{
  CLSAGFW_INITIAL_CHECK     = 0x0001,
  CLSAGFW_CHECK_AFTER_CMD   = 0x0002,
  CLSAGFW_PERIODIC_CHECK    = 0x0003,
  CLSAGFW_EXTERNAL_CHECK    = 0x0004,
  CLSAGFW_CHECK_FROM_EP     = 0x0008,
  CLSAGFW_INVALID_CHECK     = 0xffff
}clsagfw_checktype;

AGFWEXPORT clsagfw_checktype
clsagfw_get_check_type(const clsagfw_aectx *pAeCtx);

/**
 * Returns the actionName that the user has requested if the
 * current entrypoint is a RES_ACTION entrypoint.
 * Arguments :
 *   pAeCtx[IN]: context which is passed to user entry point
 *
 * API introduced in 12.1.
 */

AGFWEXPORT const oratext *
clsagfw_get_request_action_name(const clsagfw_aectx *pAeCtx);

/**
 * Returns the number of retry attempts for the current
 * command. 
 * Arguments :
 *   pAeCtx[IN]: context which is passed to user entry point
 * Return Value :
 *     0 - first attempt at command execution
 * n > 0 - command already attempted 'n' times. Partial 
 *         execution may have occurred in previous attempts.
 *
 * API introduced in 12.1.
 */

AGFWEXPORT ub4
clsagfw_get_retry_count(const clsagfw_aectx *pAeCtx);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* __CLSAGFW_H */

